/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.

	$Id: pgpFilteredSet.c,v 1.11 2001/01/25 22:11:08 jeffc Exp $
____________________________________________________________________________*/
/*
 * pgpFilteredSet.c -- Filtered variant of key sets
 */
#include "pgpConfig.h"

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <string.h>
#include <ctype.h>

#include "pgpKeyPriv.h"
#include "pgpDebug.h"
#include "pgpErrors.h"
#include "pgpTypes.h"
#include "pgpMem.h"
#include "pgpContext.h"


/* Private data for KeySet of type filter */
typedef struct FilteredSetPriv
{
	PGPKeySetRef	origSet;
	PGPFilterRef	filter;
} FilteredSetPriv;


	static PGPBoolean
sFilteredSetIsMember(
	PGPKeySetRef	keys,
	PGPKeyDBObjRef	obj )
{
	FilteredSetPriv *	priv = (FilteredSetPriv *) keys->priv;

	pgpa((
		pgpaPGPKeySetValid( keys ),
		pgpaAddrValid( priv, FilteredSetPriv )));

	return priv->origSet->isMember( priv->origSet, obj )
		&& pgpKeyDBObjMatchesFilter( priv->filter, obj );
}	

	static PGPBoolean
sFilteredSetIsEmpty(
	PGPKeySetRef	keys )
{
	FilteredSetPriv *	priv = (FilteredSetPriv *) keys->priv;
	PGPKeyDB *			kdb;
	PGPKeyDBObjRef		key;

	pgpa((
		pgpaPGPKeySetValid( keys ),
		pgpaAddrValid( priv, FilteredSetPriv )));

	if( priv->origSet->isEmpty( priv->origSet ) )
		return TRUE;

	kdb = priv->origSet->keyDB;
	for (key = kdb->firstKeyInDB; key; key = key->next)
	{
		pgpa(pgpaPGPKeyValid(key));
		if( sFilteredSetIsMember( keys, key ) )
			return FALSE;
	}

	return TRUE;
}	

	static void
sFilteredSetDestroy(
	PGPKeySet *		keys )
{
	PGPContextRef		context = PGPPeekKeySetContext( keys );
	FilteredSetPriv *	priv = (FilteredSetPriv *)keys->priv;

	if( IsntNull( priv ) )
	{
		pgpa((
			pgpaAddrValid( keys, PGPKeySet ),
			pgpaAddrValid( priv, FilteredSetPriv )));

		PGPFreeKeySet( priv->origSet );
		PGPFreeFilter( priv->filter );
		pgpContextMemFree( context, priv);
	}
}


/*
 * Create a filtered KeySet based on the specified original.  
 */
	PGPError
PGPFilterKeySet (
	PGPKeySetRef		origSet,
	PGPFilterRef		filter,
	PGPKeySetRef *		outSet )
{
	PGPContextRef		context	= PGPPeekKeySetContext( origSet );
	PGPError			err		= kPGPError_NoErr;
	FilteredSetPriv *	priv	= NULL;
	PGPKeySetRef		newSet	= kInvalidPGPKeySetRef;

	PGPValidatePtr( outSet );
	*outSet	= kInvalidPGPKeySetRef;
	PGPValidateKeySet( origSet );
	PGPValidateFilter( filter );

	pgpEnterPGPErrorFunction();

	priv = (FilteredSetPriv *)pgpContextMemAlloc( context,
												  sizeof( *priv ),
												  kPGPMemoryMgrFlags_Clear );
	if (priv == NULL)
		return kPGPError_OutOfMemory;

	err = pgpNewKeySetInternal( origSet->keyDB, &newSet );
	if ( IsPGPError( err ) )
	{
		pgpContextMemFree( context, priv );
		return err;
	}

	priv->origSet		= origSet;
	PGPIncKeySetRefCount(origSet);

	priv->filter		= filter;
	PGPIncFilterRefCount(filter);

	newSet->priv		= priv;
	newSet->isMember	= sFilteredSetIsMember;
	newSet->isEmpty		= sFilteredSetIsEmpty;
	newSet->makeUnion	= NULL;
	newSet->destroy		= sFilteredSetDestroy;

	*outSet = newSet;
	pgpAssertErrWithPtr( err, *outSet );
	return err;
}


/*__Editor_settings____

	Local Variables:
	tab-width: 4
	End:
	vi: ts=4 sw=4
	vim: si
_____________________*/
